﻿/*:
 * @target MZ
 * @plugindesc Display a customizable gauge on the screen.
 * @author Malomi
 * @url https://gendaigoyaku.com/
 * 
 * @param GaugeX
 * @type number
 * @desc The X coordinate of the gauge position. The larger the value, the more it moves to the right.
 * @default 158
 * 
 * @param GaugeY
 * @type number
 * @desc The Y coordinate of the gauge position. The larger the value, the more it moves downward.
 * @default 20
 * 
 * @param GaugeColor
 * @type string
 * @desc The color of the gauge (e.g., "red", "green", "blue", "yellow").
 * @default red
 * 
 * @param GaugeLength
 * @type number
 * @desc The length of the gauge in pixels. The larger the value, the longer the gauge.
 * @default 500
 * 
 * @param StartVisible
 * @type boolean
 * @desc Set to true if the gauge should be visible at game start.
 * @default true
 * 
 * @command GaugeFillLevel
 * @text GaugeFillLevel
 * @desc Set the gauge fill level as a percentage between 0 and 100.
 * 
 * @arg percentage
 * @type number
 * @min 0
 * @max 100
 * @desc What percentage the gauge is filled to. (0-100).
 * @default 100
 * 
 * @command ShowGauge
 * @text Show Gauge
 * @desc Display the gauge on the screen.
 * 
 * @command HideGauge
 * @text Hide Gauge
 * @desc Hide the gauge from the screen.
 * 
 * @help
 * ----------------------------------------------------------------------------
 * Gauge.js
 * ----------------------------------------------------------------------------
 * This plugin displays a gauge on the screen 
 * with customizable position, color, and initial visibility.
 * 
 * There are three types of plugin commands.
 * - GaugeFillLevel:Set the gauge fill level as a percentage between 0 and 100.
 * - ShowGauge : Display the gauge.
 * - HideGauge : Hide the gauge.
 * ----------------------------------------------------------------------------
 * License:
 * This is the MIT License. 
 * It can be used for both commercial and non-commercial purposes.
 * ----------------------------------------------------------------------------
 */

/*:ja
 * @target MZ
 * @plugindesc カスタマイズ可能なゲージを画面に表示します。
 * @author まろみ
 * @url https://gendaigoyaku.com/
 * 
 * @param GaugeX
 * @type number
 * @desc ゲージ位置のX座標。値が大きいほど右へ移動します。0が左の端です。
 * @default 158
 * 
 * @param GaugeY
 * @type number
 * @desc ゲージ位置のY座標。値が大きいほど下へ移動します。0が上の端です。
 * @default 20
 * 
 * @param GaugeColor
 * @type string
 * @desc ゲージの色を指定できます。半角アルファベットで、英語で書いてね (例, "red", "green", "blue", "yellow").
 * @default red
 * 
 * @param GaugeLength
 * @type number
 * @desc ゲージの長さ（ピクセル数）。値が大きいほどゲージが長くなります。
 * @default 500
 * 
 * @param StartVisible
 * @type boolean
 * @desc ゲーム開始時にゲージを表示する場合はtrueに設定します。
 * @default true
 * 
 * @command GaugeFillLevel
 * @text ゲージの増減
 * @desc ゲージが何パーセントたまっているかを設定します。
 * 
 * @arg percentage
 * @type number
 * @min 0
 * @max 100
 * @desc ゲージのたまり具合をパーセントで指定します。0から100までの数字で設定してね。
 * @default 100
 * 
 * @command ShowGauge
 * @text ゲージを表示する
 * @desc ゲージを表示します。
 * 
 * @command HideGauge
 * @text ゲージを非表示にする
 * @desc ゲージを非表示にします。
 * 
 * @help
 * ----------------------------------------------------------------------------
 * Gauge.js
 * ----------------------------------------------------------------------------
 * ゲージを表示するプラグインです。
 * 表示する位置、ゲージの長さ、色を設定できます。
 * 表示と非表示を切り替えられます。
 * 
 * プラグインコマンド.
 * - ゲージの増減 : ゲーム中、任意にゲージを増減できます。
 * - ゲージを表示する : ゲージを表示します。
 * - ゲージを非表示にする : ゲージを非表示にします。
 * ----------------------------------------------------------------------------
 * 利用規約:
 * 商用・非商用問わず利用可能です。
 * ----------------------------------------------------------------------------
 */

(() => {
    'use strict';

    const parameters = PluginManager.parameters('Gauge');
    const gaugeX = Number(parameters['GaugeX']) || 158;
    const gaugeY = Number(parameters['GaugeY']) || 20;
    const gaugeColor = parameters['GaugeColor'] || 'red';
    const gaugeLength = Number(parameters['GaugeLength']) || 500;
    const startVisible = parameters['StartVisible'] === 'true';

    let myValue = 1;
    let isGaugeVisible = startVisible;

    PluginManager.registerCommand('Gauge', 'GaugeFillLevel', args => {
        myValue = Number(args.percentage) / 100;
    });

    PluginManager.registerCommand('Gauge', 'ShowGauge', () => {
        isGaugeVisible = true;
    });

    PluginManager.registerCommand('Gauge', 'HideGauge', () => {
        isGaugeVisible = false;
    });

    class GaugeSprite extends Sprite {
        constructor() {
            super(new Bitmap(gaugeLength + 20, 20));
            this.x = gaugeX;
            this.y = gaugeY;
            this._value = 1;
            this.refresh();
        }
        set value(val) {
            this._value = val;
            this.refresh();
        }
        get value() {
            return this._value;
        }
        refresh() {
            this.bitmap.clear();
            this.bitmap.fillRect(0, 0, gaugeLength, 20, 'black');
            this.bitmap.fillRect(2, 2, (gaugeLength - 4) * this._value, 16, gaugeColor);
        }
    }

    const _Scene_Map_createSpriteset = Scene_Map.prototype.createSpriteset;
    Scene_Map.prototype.createSpriteset = function () {
        _Scene_Map_createSpriteset.call(this);
        this._gaugeSprite = new GaugeSprite();
        this._gaugeSprite.visible = isGaugeVisible;
        this.addChild(this._gaugeSprite);
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        if (this._gaugeSprite) {
            this._gaugeSprite.value = myValue;
            this._gaugeSprite.visible = isGaugeVisible;
        }
    };
})();
